import numpy as np
import scipy.ndimage
from matplotlib import pyplot as plt
import matplotlib
from matplotlib.patches import Polygon
from matplotlib.collections import PatchCollection
import matplotlib.gridspec as gridspec
from cycler import cycler


######################################################################################################################################################
#   Code for producing Figure 8 in the main text.
#   The following data is used in this figure:
#   -- theoryStDev_vs_muc.txt, the standard deviation of the step distribution as a function of the inter-leg constraint strength,
#                                   with angle preference \thata_p = 83 degrees (the value used throughout the manuscript)
#   -- theoryStDev_vs_muc_thetap65.txt, the standard deviation of the step distribution as a function of the inter-leg constraint strength,
#                                   with angle preference \thata_p = 65 degrees
#   -- theoryStDev_vs_muc_thetap95.txt, the standard deviation of the step distribution as a function of the inter-leg constraint strength,
#                                   with angle preference \thata_p = 95 degrees

#   -- theoryBindingTime_vs_muc.txt, the mean binding time as a function of the inter-leg constraint strength, with angle
#                                    preference \thata_p = 81 degrees (the value used throughout the manuscript)
#   -- theoryBindingTime_vs_muc_thetap65.txt, the mean binding time as a function of the inter-leg constraint strength, with angle
#                                    preference \thata_p = 65 degrees
#   -- theoryBindingTime_vs_muc_thetap95.txt, the mean binding time as a function of the inter-leg constraint strength, with angle
#                                    preference \thata_p = 95 degrees (the value used throughout the manuscript)
#
######################################################################################################################################################



color = plt.cm.YlOrBr(np.linspace(0.4,1.,3)) # This returns RGBA; convert:

matplotlib.rcParams['axes.prop_cycle'] = cycler(color=color)

axisLabelSize = 28
lw = 2.5
msize = 7
legendTxt = 19
mkWidth = 2
handle = 2.75
tickLen=8
tickLabel=20

params = {'backend': 'ps'}

plt.rcParams.update(params)

muc, stdev  = np.transpose(np.loadtxt('theory_constraint_significance/theoryStDev_vs_muc.txt'))
_, stdev65  = np.transpose(np.loadtxt('theory_constraint_significance/theoryStDev_vs_muc_thetap65.txt'))
_, stdev95  = np.transpose(np.loadtxt('theory_constraint_significance/theoryStDev_vs_muc_thetap95.txt'))
_, bind  = np.transpose(np.loadtxt('theory_constraint_significance/theoryBindingTime_vs_muc.txt'))
_, bind65  = np.transpose(np.loadtxt('theory_constraint_significance/theoryBindingTime_vs_muc_thetap65.txt'))
_, bind95  = np.transpose(np.loadtxt('theory_constraint_significance/theoryBindingTime_vs_muc_thetap95.txt'))


fig, ax1 = plt.subplots(figsize=(10, 10*(np.sqrt(5)-1.0)/2.0))

plt.tick_params('both',length=tickLen,labelsize=tickLabel)


color = 'tab:red'
ax1.set_xlabel(r'$\mu_c$',size=axisLabelSize)
ax1.set_ylabel(r'Standard Deviation [nm]',size=axisLabelSize, labelpad=10)
l1 = ax1.plot(muc, stdev65, linewidth=lw, label=r'    $\,65\degree$    ')
l2 = ax1.plot(muc, stdev, linewidth=lw, label=r'    $\,83\degree$    ')
l3 = ax1.plot(muc, stdev95, linewidth=lw, label=r'    $\,95\degree$    ')
ax1.tick_params('both',length=tickLen,labelsize=tickLabel)
ax1.locator_params(nbins=5)


ax2 = ax1.twinx()  # instantiate a second axes that shares the same x-axis

color = 'tab:blue'
ax2.set_ylabel(r'Mean Binding Time [ms]',size=axisLabelSize, labelpad=10)  # we already handled the x-label with ax1
l4 = ax2.plot(muc, bind65, '--', linewidth=lw, label=' ')
l5 = ax2.plot(muc, bind, '--', linewidth=lw, label=' ')
l6 = ax2.plot(muc, bind95, '--', linewidth=lw, label=' ')
ax2.tick_params('both',length=tickLen,labelsize=tickLabel)
ax2.set_ylim([1.25,3.])
ax2.locator_params(nbins=5)

lns = l1+l2+l3+l4+l5+l6
labs = [l.get_label() for l in lns]
leg = plt.legend(lns, labs, frameon=False, fontsize=legendTxt, handlelength=handle, title=r"$\sigma_\mathrm{step}\,\,$    $\theta_p$       $\,t_\mathrm{T}$",ncol=2,columnspacing=0,handletextpad=0)
plt.setp(leg.get_title(),fontsize=23)
leg._legend_box.align = "left"

fig.tight_layout()


plt.savefig('constraint_significance.pdf',bbox_inches='tight')

plt.show()